<?php
/**
 * Google Client Helper Functions for WHMCS (CDN Version)
 * 
 * This version uses CDN to load Google API - NO vendor folder needed!
 */

require_once 'whmcs-google-config.php';

/**
 * Get Google Auth URL (Manual approach without library)
 * 
 * @return string
 */
function getGoogleAuthUrl() {
    $params = [
        'client_id' => GOOGLE_CLIENT_ID,
        'redirect_uri' => GOOGLE_REDIRECT_URI,
        'response_type' => 'code',
        'scope' => 'email profile',
        'access_type' => 'offline',
        'prompt' => 'consent'
    ];
    
    return 'https://accounts.google.com/o/oauth2/v2/auth?' . http_build_query($params);
}

/**
 * Exchange authorization code for access token
 * 
 * @param string $code
 * @return array|false
 */
function exchangeCodeForToken($code) {
    $token_url = 'https://oauth2.googleapis.com/token';
    
    $post_data = [
        'code' => $code,
        'client_id' => GOOGLE_CLIENT_ID,
        'client_secret' => GOOGLE_CLIENT_SECRET,
        'redirect_uri' => GOOGLE_REDIRECT_URI,
        'grant_type' => 'authorization_code'
    ];
    
    $ch = curl_init($token_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($post_data));
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code == 200) {
        return json_decode($response, true);
    }
    
    return false;
}

/**
 * Get user info from Google
 * 
 * @param string $access_token
 * @return array|false
 */
function getGoogleUserInfo($access_token) {
    $userinfo_url = 'https://www.googleapis.com/oauth2/v2/userinfo';
    
    $ch = curl_init($userinfo_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $access_token
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code == 200) {
        return json_decode($response, true);
    }
    
    return false;
}

/**
 * Encrypt password using bcrypt (WHMCS compatible)
 * 
 * @param string $password
 * @return string
 */
function whmcsEncryptPassword($password) {
    return password_hash($password, PASSWORD_BCRYPT);
}

/**
 * Generate random secure password
 * 
 * @param int $length
 * @return string
 */
function generateRandomPassword($length = 16) {
    return bin2hex(random_bytes($length / 2));
}

/**
 * Set WHMCS Client Session
 * 
 * @param int $client_id
 * @param string $password_hash
 */
function setWHMCSSession($client_id, $password_hash) {
    $_SESSION['uid'] = $client_id;
    $_SESSION['upw'] = $password_hash;
    $_SESSION['cid'] = '';
    $_SESSION['adminid'] = 0;
}

/**
 * Log activity in WHMCS
 * 
 * @param int $client_id
 * @param string $description
 */
function logWHMCSActivity($client_id, $description) {
    global $whmcs_pdo;
    
    try {
        $stmt = $whmcs_pdo->prepare("
            INSERT INTO tblactivitylog 
            (date, description, user, userid, ipaddr) 
            VALUES (NOW(), ?, '', ?, ?)
        ");
        
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? '';
        $stmt->execute([$description, $client_id, $ip_address]);
    } catch (Exception $e) {
        // Silently fail - logging shouldn't break the flow
        error_log('WHMCS Activity Log Error: ' . $e->getMessage());
    }
}
?>